// IndexedDB queue for offline submissions + user filter + badge support
const DB_NAME = 'formbuilder-offline';
const STORE = 'pendingSubmissions';

function idb() {
  return new Promise((resolve, reject) => {
    const open = indexedDB.open(DB_NAME, 1);
    open.onupgradeneeded = () => {
      const db = open.result;
      if (!db.objectStoreNames.contains(STORE)) db.createObjectStore(STORE, { keyPath: 'id' });
    };
    open.onsuccess = () => resolve(open.result);
    open.onerror = () => reject(open.error);
  });
}

function currentUsername(){
  try{
    if (window.CURRENT_USER && (CURRENT_USER.username || CURRENT_USER.display_name)) {
      return CURRENT_USER.username || CURRENT_USER.display_name;
    }
  }catch(e){}
  return null;
}

async function queueAdd(payload) {
  const db = await idb();
  const tx = db.transaction(STORE, 'readwrite');
  tx.objectStore(STORE).put(payload);
  await (tx.complete || new Promise(res => tx.oncomplete = res));
  dispatchEvent(new Event('queuechange'));
}
export async function queueAll() {
  const db = await idb();
  const tx = db.transaction(STORE, 'readonly');
  return new Promise(res => {
    const req = tx.objectStore(STORE).getAll();
    req.onsuccess = () => res(req.result || []);
  });
}
async function queueDel(id) {
  const db = await idb();
  const tx = db.transaction(STORE, 'readwrite');
  tx.objectStore(STORE).delete(id);
  await (tx.complete || new Promise(res => tx.oncomplete = res));
  dispatchEvent(new Event('queuechange'));
}

export async function pendingCount(onlyCurrentUser=true){
  const all = await queueAll();
  if (!onlyCurrentUser) return all.length;
  const user = currentUsername();
  if (!user) return all.length;
  return all.filter(p => (p.user||null) === user).length;
}

export async function attachOfflineSubmit(form) {
  form.addEventListener('submit', async (ev) => {
    ev.preventDefault();
    const data = new FormData(form);
    const id = Date.now() + '-' + (Math.random()*1e6|0);
    const payload = { id, url: form.action, method: form.method || 'POST', ts: Date.now() };

    // capture user + form_id if present
    payload.user = currentUsername();
    if (!payload.user && data.has('username')) payload.user = data.get('username');
    if (data.has('form_id')) payload.form_id = data.get('form_id');

    const plain = {}; data.forEach((v,k) => { plain[k] = v; });
    payload.body = plain;

    async function sendOnline() {
      const res = await fetch(payload.url, { method: 'POST', body: new FormData(form) });
      if (!res.ok) throw new Error('HTTP '+res.status);
      return res;
    }

    if (navigator.onLine) {
      try { await sendOnline(); location.reload(); return; }
      catch (e) { /* queue below */ }
    }
    await queueAdd(payload);
    alert('Sem conexão. O registro foi salvo na Caixa de Saída e será enviado quando a internet voltar.');
    try { if ('serviceWorker' in navigator && 'sync' in (await navigator.serviceWorker.ready)) {
      (await navigator.serviceWorker.ready).sync.register('form-sync');
    }} catch(e) {}
  });
}

export async function trySync() {
  if (!navigator.onLine) return;
  const pend = await queueAll();
  for (const p of pend) {
    try {
      const fd = new FormData();
      Object.entries(p.body||{}).forEach(([k,v]) => fd.append(k, v));
      const res = await fetch(p.url, { method:'POST', body:fd });
      if (res.ok) await queueDel(p.id);
    } catch(e) { /* keep queued */ }
  }
  dispatchEvent(new Event('queuechange'));
}

export async function renderOutbox(container) {
  const all = await queueAll();
  const user = currentUsername();
  const items = user ? all.filter(p => (p.user||null) === user) : all;
  if (!items.length) { container.innerHTML = '<div class="card">Nenhuma pendência na Caixa de Saída.</div>'; return; }
  const rows = items.map(p => `
    <tr>
      <td>${new Date(p.ts).toLocaleString()}</td>
      <td>${p.url}</td>
      <td>${p.form_id || ''}</td>
      <td>${p.user || ''}</td>
      <td><button class="btn" data-id="${p.id}">Enviar agora</button></td>
      <td><button class="btn is-danger" data-del="${p.id}">Excluir</button></td>
    </tr>
  `).join('');
  container.innerHTML = `
    <table class="table">
      <thead><tr><th>Data</th><th>Destino</th><th>Form</th><th>Usuário</th><th colspan="2">Ações</th></tr></thead>
      <tbody>${rows}</tbody>
    </table>`;
  container.querySelectorAll('[data-id]').forEach(btn => {
    btn.onclick = async () => { await trySync(); renderOutbox(container); };
  });
  container.querySelectorAll('[data-del]').forEach(btn => {
    btn.onclick = async () => { await queueDel(btn.getAttribute('data-del')); renderOutbox(container); };
  });
}
window.addEventListener('online', trySync);
